-- BeamGarage Extension
-- Export and import vehicle configurations for BeamGarage.com

local M = {}

-- Log helper
local function logInfo(msg)
  log('I', 'BeamGarage', msg)
end

local function logError(msg)
  log('E', 'BeamGarage', msg)
end

-- Strip unnecessary metadata from parts tree to reduce export size
-- Keeps only: chosenPartName (as "part"), children (recursively cleaned)
local function cleanPartsTree(node)
  if not node or type(node) ~= 'table' then
    return nil
  end

  local cleaned = {}

  -- Keep the chosen part name (shortened key)
  if node.chosenPartName and node.chosenPartName ~= '' then
    cleaned.part = node.chosenPartName
  end

  -- Recursively clean children
  if node.children and type(node.children) == 'table' then
    local cleanedChildren = {}
    local hasChildren = false
    for key, child in pairs(node.children) do
      local cleanedChild = cleanPartsTree(child)
      if cleanedChild and next(cleanedChild) then
        cleanedChildren[key] = cleanedChild
        hasChildren = true
      end
    end
    if hasChildren then
      cleaned.children = cleanedChildren
    end
  end

  return cleaned
end

-- Flatten compact parts tree to a flat slot -> partName dictionary
-- BeamNG expects: { "slot_name": "part_name", ... }
local function flattenPartsTree(node, result)
  result = result or {}

  if not node or type(node) ~= 'table' then
    return result
  end

  -- Process children - the key is the slot name, child.part is the part name
  if node.children and type(node.children) == 'table' then
    for slotName, child in pairs(node.children) do
      if child.part then
        result[slotName] = child.part
      end
      -- Recursively process nested children
      flattenPartsTree(child, result)
    end
  end

  return result
end

-- Called when extension is loaded
M.onExtensionLoaded = function()
  logInfo('BeamGarage extension loaded')
end

-- Export current vehicle configuration
-- Retrieves config via core_vehicle_partmgmt and sends to UI
M.exportConfig = function()
  -- Get current player vehicle
  local vehicle = be:getPlayerVehicle(0)

  if not vehicle then
    logError('No vehicle spawned')
    guihooks.trigger('BeamGarageError', { message = 'No vehicle spawned. Please spawn a vehicle first.' })
    return
  end

  -- Get vehicle info
  local model = vehicle:getField('JBeam', '')
  local vehicleId = vehicle:getId()

  logInfo('Exporting config for model: ' .. model .. ', id: ' .. vehicleId)

  -- Get config via core_vehicle_partmgmt
  local partmgmt = extensions.core_vehicle_partmgmt
  if not partmgmt then
    logError('core_vehicle_partmgmt not available')
    guihooks.trigger('BeamGarageError', { message = 'Part management not available.' })
    return
  end

  -- Get current parts config
  local partsConfig = partmgmt.getConfig()
  if not partsConfig then
    logError('Failed to get parts config')
    guihooks.trigger('BeamGarageError', { message = 'Failed to get vehicle configuration.' })
    return
  end

  -- Log keys for debugging
  local keys = {}
  for k, _ in pairs(partsConfig) do
    table.insert(keys, tostring(k))
  end
  logInfo('Got parts config, keys: ' .. table.concat(keys, ', '))

  -- Build export config - partsConfig may have parts or partsTree
  -- Clean the parts tree to remove unnecessary metadata (reduces size by ~80%)
  local rawParts = partsConfig.parts or partsConfig.partsTree or {}
  local cleanedParts = cleanPartsTree(rawParts)

  local exportConfig = {
    format = 2,
    model = model,
    parts = cleanedParts or {},
    vars = partsConfig.vars or {},
    paints = partsConfig.paints or {}
  }

  -- Encode to JSON for transfer
  local jsonString = jsonEncode(exportConfig)

  logInfo('Config export ready, sending to UI')

  -- Send to UI
  guihooks.trigger('BeamGarageExportReady', {
    config = jsonString,
    model = model
  })
end

-- Base64 decode helper
local function base64Decode(data)
  local b = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/'
  data = string.gsub(data, '[^'..b..'=]', '')
  return (data:gsub('.', function(x)
    if (x == '=') then return '' end
    local r, f = '', (b:find(x) - 1)
    for i = 6, 1, -1 do r = r .. (f % 2^i - f % 2^(i-1) > 0 and '1' or '0') end
    return r
  end):gsub('%d%d%d?%d?%d?%d?%d?%d?', function(x)
    if (#x ~= 8) then return '' end
    local c = 0
    for i = 1, 8 do c = c + (x:sub(i, i) == '1' and 2^(8-i) or 0) end
    return string.char(c)
  end))
end

-- Import configuration from Base64 encoded JSON string
M.importConfigBase64 = function(base64String)
  logInfo('importConfigBase64 called, base64 length: ' .. (base64String and string.len(base64String) or 0))

  if not base64String or base64String == '' then
    logError('No config provided')
    guihooks.trigger('BeamGarageImportResult', {
      success = false,
      message = 'No configuration data provided.'
    })
    return
  end

  -- Decode Base64
  logInfo('Decoding Base64...')
  local jsonString = base64Decode(base64String)
  logInfo('Decoded JSON length: ' .. string.len(jsonString))
  logInfo('First 100 chars: ' .. string.sub(jsonString, 1, 100))

  -- Call the main import function
  logInfo('Calling importConfig...')
  M.importConfig(jsonString)
  logInfo('importConfig returned')
end

-- Import configuration from JSON string
-- Applies the config to the current vehicle
M.importConfig = function(jsonString)
  if not jsonString or jsonString == '' then
    logError('No config provided')
    guihooks.trigger('BeamGarageImportResult', {
      success = false,
      message = 'No configuration data provided.'
    })
    return
  end

  -- Parse the JSON
  local config = jsonDecode(jsonString)
  if not config then
    logError('Failed to parse import JSON')
    guihooks.trigger('BeamGarageImportResult', {
      success = false,
      message = 'Invalid JSON format. Please check the configuration data.'
    })
    return
  end

  -- Validate required fields
  if not config.model then
    logError('Config missing model field')
    guihooks.trigger('BeamGarageImportResult', {
      success = false,
      message = 'Configuration missing required "model" field.'
    })
    return
  end

  if not config.parts then
    logError('Config missing parts field')
    guihooks.trigger('BeamGarageImportResult', {
      success = false,
      message = 'Configuration missing required "parts" field.'
    })
    return
  end

  -- Get current player vehicle
  local vehicle = be:getPlayerVehicle(0)

  if not vehicle then
    logError('No vehicle spawned for import')
    guihooks.trigger('BeamGarageImportResult', {
      success = false,
      message = 'No vehicle spawned. Please spawn a vehicle first.'
    })
    return
  end

  -- Get current vehicle model
  local currentModel = vehicle:getField('JBeam', '')

  -- Check if models match
  local modelMismatch = false
  if currentModel ~= config.model then
    logInfo('Model mismatch: current=' .. currentModel .. ', config=' .. config.model)
    modelMismatch = true
  end

  -- Prepare the config for applying
  -- Flatten the parts tree to a flat slot -> partName dictionary
  -- BeamNG expects: { "slot_name": "part_name", ... }
  local flatParts = flattenPartsTree(config.parts)

  -- Also add the root part if present
  if config.parts and config.parts.part then
    flatParts[config.model] = config.parts.part
  end

  local applyConfig = {
    format = config.format or 2,
    model = config.model,
    parts = flatParts,
    vars = config.vars or {},
    paints = config.paints or {}
  }

  -- Log the number of parts for debugging
  local partCount = 0
  for _ in pairs(flatParts) do partCount = partCount + 1 end
  logInfo('Applying config for model: ' .. config.model .. ' with ' .. partCount .. ' parts')

  -- Apply config by respawning the vehicle with the full configuration
  -- This is the most reliable method (used by BeamMP) to apply parts, vars, and paints
  local success, err = pcall(function()
    -- Serialize the config and respawn the vehicle with it
    local configString = serialize(applyConfig)
    logInfo('Respawning vehicle with config, length: ' .. string.len(configString))
    vehicle:respawn(configString)
  end)

  if not success then
    logError('Vehicle respawn failed: ' .. tostring(err))
    guihooks.trigger('BeamGarageImportResult', {
      success = false,
      message = 'Failed to apply config: ' .. tostring(err)
    })
    return
  end

  logInfo('Vehicle respawned with full config (parts + vars + paints)')

  logInfo('Config import command queued')

  -- Send success result
  local message = 'Configuration applied!'
  if modelMismatch then
    message = 'Configuration applied. Note: Config is for ' .. config.model .. ' but current vehicle is ' .. currentModel .. '. Some parts may not apply correctly.'
  end

  logInfo('About to trigger BeamGarageImportResult with success=true')
  logInfo('Message: ' .. message)
  guihooks.trigger('BeamGarageImportResult', {
    success = true,
    message = message,
    modelMismatch = modelMismatch
  })
  logInfo('BeamGarageImportResult triggered')
end

return M
